<?php

/*
 * Radium Framework Core - A WordPress theme development framework.
 * 
 * This file is a part of the RadiumFramework core.
 * Please be extremely cautious editing this file. 
 * Modifying the contents of this file can be a poor life decision if you don't know what you're doing.
 * 
 * @category RadiumFramework
 * @package  MetroCorp WP
 * @author   Franklin M Gitonga
 * @link     http://radiumthemes.com/
 */

// Exit if accessed directly
if ( !defined( 'ABSPATH' ) ) exit;

//Load Main framework - very important
require_once('framework/radium.php');

/**
 * The Radium_Framework class launches the framework.  It's the organizational structure behind the entire theme. 
 * This class should be loaded and initialized before anything else within the theme is called to properly use 
 * the framework.  
 *
 * Initializes the framework by doing some basic things like defining constants
 * and loading framework components 
 * 
 * The framework is contained in "framework/" while customizable theme files are contained in "includes/"
 * @since 2.1.0
 */
 
if ( class_exists( 'Radium_Framework' ) ) : //check if the Radium Framework class exists

	class Radium_Theme_Framework extends Radium_Framework {
	
		/** Singleton *************************************************************/
	
		/**
		 * @var radium_framework The one true radium framework
		 */
		private static $instance;
		
		private function __construct() { parent::instance(); }
			
		/**
		 * Main radium Instance
		 *
		 * Please load it only one time
		 * For this, we thank you
		 *
		 * Insures that only one instance of the radium framework exists in memory at any one
		 * time. Also prevents needing to define globals all over the place.
		 *
		 * @since 2.1.0
		 * @static var array $instance
		 * @uses radium_framework::setup_globals() Setup the globals needed
		 * @uses radium_framework::includes() Include the required files
		 * @uses radium_framework::setup_actions() Setup the hooks and actions
		 * @see radium_framework()
		 * @return The one true radium framework
		 */
		public static function instance() {
			if ( ! isset( self::$instance ) ) {
				self::$instance = new self();
				self::$instance->setup_globals();
				self::$instance->includes();
			}
			return self::$instance;
		}
				
		/*
		 * Setup the config array for which features the 
		 * theme supports. This can easily be filtered 
		 * giving you a chance to disable/enable the theme's various features.
		 *
		 * set each feature to true or false
		 * 
		 * radium_feature_setup 
		 *
		 * @since 2.0.0
		 */
		public function feature_setup() {
		
			$args = array(
			
				'primary' 	=> array(
					'breadcrumbs'		=> true, 
					'builder'			=> true, 
					'meta'				=> true,
					'megamenu'			=> true,
					'portfolio'			=> true,
					'responsive' 		=> true,			 
					'shortcodes'		=> true, 
					'sliders' 			=> true,			 
					'skins' 			=> true,
					'widgets'			=> true, 
				),
								
				'comments' 	=> array(
					'pages'				=> false,  //show comments on pages
					'portfolio'			=> false,  //show comments on single portfolios
					'posts'				=> true,   //show comments on single posts
				),
				
				'admin' => array(
					'options'			=> true,			// Entire Admin presence
					'builder'			=> true,			// Layouts Builder page
				),
								
				'meta' => array(
					'hijack_atts'		=> true,			// Hijack and modify "Page Attributes"
					'builder'			=> true,			// Layouts Builder page
				)
				 
			);
			return apply_filters( 'radium_theme_config_args', $args );
		}
			
		/** Private Methods *******************************************************/
		
		/**
		 * Set some smart defaults to class variables. Allow some of them to be
		 * filtered to allow for early overriding.
		 *
		 * @since 1.0
		 * @access private
		 * @uses get_template_directory() To generate theme path
		 * @uses get_template_directory_uri() To generate bbPress theme url
		 * @uses apply_filters() calls various filters
		 */
		
		private function setup_globals() {
			
		    /** Versions **********************************************************/
		    	
			/** Define Theme Info Constants */
			$theme 						= wp_get_theme(); //Get Theme data (WP 3.4+)
			$this->theme_version     	= $theme->version; // Theme version
			$this->theme_framework_version 	= '2.1.2';   // framework version
			
			/** Paths *************************************************************/
			
			// Setup some base path, name and URL information
			$this->theme_title  		= apply_filters( 'radium_theme_title',  		$theme->name ); //or $theme->title
			$this->theme_slug  			= apply_filters( 'radium_theme_slug',  			get_template() );
			
			$this->theme_dir 			= apply_filters( 'radium_theme_dir_path',  		get_template_directory() );
			$this->theme_url 			= apply_filters( 'radium_theme_dir_url',   		get_template_directory_uri() );
			
			//Setup Child theme path and URL
			$this->child_theme_dir 		= apply_filters( 'radium_child_theme_dir_path', get_stylesheet_directory() );
			$this->child_theme_url 		= apply_filters( 'radium_child_theme_dir_url',  get_stylesheet_directory_uri() );
			
			//Setup theme Options name - it's not recommended that you change this, if you do you will looses theme option settings
			$this->theme_option_name 	= $this->theme_slug . '_options';   // Theme_options name
			
		  	/*----------------------------------------------------*/
		  	/* Define General Constants */
		  	/*----------------------------------------------------*/
		 
		 	/** Define Directory Location Constants (These Constants make moving directories and files around very easy) */
			$this->theme_assets_dir 	= apply_filters( 'radium_theme_assets_dir', 	$this->theme_dir . 		'/assets' );
			$this->theme_images_dir 	= apply_filters( 'radium_theme_images_dir', 	$this->theme_dir . 		'/assets/images' );
			$this->theme_includes_dir 	= apply_filters( 'radium_theme_includes_dir',	$this->theme_dir . 		'/includes' );
			$this->theme_js_dir 		= apply_filters( 'radium_theme_js_dir',  		$this->theme_dir . 		'/assets/js' );
			$this->theme_css_dir 		= apply_filters( 'radium_theme_css_dir',  		$this->theme_dir . 		'/assets/css' );
			$this->theme_functions_dir 	= apply_filters( 'radium_theme_functions_dir',  $this->theme_includes_dir . 	'/functions' );
			$this->theme_content_dir 	= apply_filters( 'radium_theme_content_dir',  	$this->theme_includes_dir . 	'/content' );
			$this->lang_dir   			= apply_filters( 'radium_lang_dir', trailingslashit( $this->theme_includes_dir . '/languages' ) );
			$this->theme_cpt_dir   		= apply_filters( 'radium_theme_cpt_dir', trailingslashit( $this->theme_includes_dir . '/custom-post-types' ) );
		
			/** Define Url Location Constants (These Constants make moving directories and files around very easy) */
			$this->theme_assets_url 	= apply_filters( 'radium_theme_assets_url',  	$this->theme_url . 		'/assets' );
		 	$this->theme_styles_url 	= apply_filters( 'radium_theme_styles_url',  	$this->theme_url . 		'/assets/styles' );
		 	$this->theme_images_url 	= apply_filters( 'radium_theme_images_url',  	$this->theme_url . 		'/assets/images' );
			$this->theme_includes_url 	= apply_filters( 'radium_theme_includes_url',  	$this->theme_url . 		'/includes' );
			$this->theme_js_url 		= apply_filters( 'radium_theme_js_url',  		$this->theme_url . 		'/assets/js' );
			$this->theme_css_url 		= apply_filters( 'radium_theme_css_url',  		$this->theme_url . 		'/assets/css' );
			$this->theme_functions_url 	= apply_filters( 'radium_theme_functions_url',  $this->theme_includes_url . 	'/functions' );
			$this->theme_cpt_url  		= apply_filters( 'radium_theme_cpt_url', 		$this->theme_includes_url . 	'/custom-post-types' );
			
			
			/*----------------------------------------------------*/
			/* Define Admin Constants */
			/*----------------------------------------------------*/
		
			/** Define Framework Directory Location Constants ***************************/
			$this->theme_framework_dir 			= apply_filters( 'radium_theme_framework_dir',  		$this->theme_dir . '/framework' );
			$this->theme_framework_images_dir 	= apply_filters( 'radium_theme_framework_images_dir',  	$this->theme_dir . '/framework/assets/images' );
			$this->theme_framework_css_dir 		= apply_filters( 'radium_theme_framework_css_dir',  	$this->theme_dir . '/framework/assets/css' );
			$this->theme_framework_js_dir 		= apply_filters( 'radium_theme_framework_js_dir',  		$this->theme_dir . '/framework/assets/js' );
		 		
			/** Define Framework URL Location Constants *********************************/
			$this->theme_framework_url 			= apply_filters( 'radium_theme_framework_url',  		$this->theme_url . '/framework' );
			$this->theme_framework_images_url 	= apply_filters( 'radium_theme_framework_images_url',  	$this->theme_url . '/framework/assets/images' );
			$this->theme_framework_css_url 		= apply_filters( 'radium_theme_framework_css_url',  	$this->theme_url . '/framework/assets/css' );
			$this->theme_framework_js_url 		= apply_filters( 'radium_theme_framework_js_url',  		$this->theme_url . '/framework/assets/js' );
			
			// Constants for the theme name, folder and remote XML url
			$this->theme_framework_site_url = apply_filters( 'radium_theme_framework_url', 		'http://radiumthemes.com/framework/' );
			$this->theme_docs_url 			= apply_filters( 'radium_theme_docs_url', 			'http://support.radiumthemes.com/knowledgebase' );
			$this->theme_support_url 		= apply_filters( 'radium_theme_support_url', 		'http://support.radiumthemes.com' );
	
			//presstrend
			$this->theme_presstrend 		= apply_filters( 'radium_theme_presstrend', 		true ); //Enable presstrend (true or false)
			$this->theme_presstrend_key 	= apply_filters( 'radium_theme_presstrend_key', 	'sxn0jeama6fng7fjz5xsuw9ou3ysu7cdm4xc' ); // Radium auth
			$this->theme_presstrend_auth 	= apply_filters( 'radium_theme_presstrend_auth', 	'u1d4xrupivnlzurlf1vdmccn5hogoake9' ); // theme auth
				
			/* DEV Mode */
			$this->theme_dev_mode = apply_filters( 'radium_theme_dev_mode', false ); //Dev Mode (true or false)
			
			/**  Modular Contants **************************************/
			 
			$this->theme_shortcodes_dir = apply_filters( 'radium_theme_shortcodes_dir',  $this->theme_includes_dir . '/shortcodes' );
			$this->theme_shortcodes_url = apply_filters( 'radium_theme_shortcodes_url',  $this->theme_includes_url . '/shortcodes' );
		
			$this->theme_widgets_dir 	= apply_filters( 'radium_theme_widgets_dir',  $this->theme_includes_dir . '/widgets' );
		 	$this->theme_widgets_url 	= apply_filters( 'radium_theme_widgets_url',  $this->theme_includes_url . '/widgets' );
			
			/* Slider Relative Location (used by radium slide engine) */
			$this->theme_slider_location 		= apply_filters( 'radium_theme_slider_location', 'includes/sliders' );
				
			$this->theme_forms_dir 		= apply_filters( 'radium_theme_forms_dir',  $this->theme_framework_dir . '/forms' );
			$this->theme_forms_url 		= apply_filters( 'radium_theme_forms_url',  $this->theme_framework_url . '/forms' );
			
			/** Misc **************************************************************/
			
			$this->domain         = 'radium';      				// Unique identifier for retrieving translated strings
			$this->extend         = new stdClass(); 			// Plugins add data here
			$this->options        = get_option( $this->theme_option_name ); // get theme options so we don't run it all the time 
			$this->errors         = new WP_Error(); 			// Feedback
			
		}
		 
		 	
		/**
		 * Loads all the framework files and features.
		 *
		 * The radium_pre_framework action hook is called before any of the files are
		 * required().
		 *
		 * @since 1.0.0
		 */
		public function includes() {
				
			/*------------------------------------------------------------------------------------
			// Load General Functions (these are important - and are needed in the frontend and backend - don't disable)
			------------------------------------------------------------------------------------*/
			require( $this->theme_framework_dir . '/core/core-functions.php' );
			
			require( $this->theme_functions_dir . '/theme-setup.php' );
			require( $this->theme_functions_dir . '/theme-functions.php' );
						
			if( $this->theme_supports( 'primary', 'meta' ) )
				require_once ($this->theme_framework_dir . '/metaboxes/metaboxes-init.php'); //metaboxes Engine
			//End General Functions
			
			/* Load Navigation Tools */
			include( $this->theme_functions_dir . '/navigation/main-menu/menu-init.php' );
			include( $this->theme_functions_dir . '/navigation/main-menu/admin-menu-walker.php' );		
			
			
			/*------------------------------------------------------------------------------------
			// Load the Shortcodes
			-------------------------------------------------------------------------------------*/
			if( $this->theme_supports( 'primary', 'shortcodes' ) )
				require( $this->theme_shortcodes_dir . '/shortcodes-init.php');
				
			/*------------------------------------------------------------------------------------
			// Load the Layout Builder
			 -------------------------------------------------------------------------------------*/
			if( $this->theme_supports('primary', 'builder') )
				include($this->theme_framework_dir . '/builder/builder-init.php');
			
			/*------------------------------------------------------------------------------------
			// Load the Slider Engines
			 ------------------------------------------------------------------------------------*/
			if( $this->theme_supports( 'primary', 'sliders' ) )
				include( $this->theme_framework_dir . '/slider/slider-init.php' );
				
			if( $this->theme_supports( 'primary', 'sliders' )  && $this->is_plugin_active('revslider/revslider.php') )
				include( $this->theme_includes_dir . '/extensions/revolution-slider.php' );
			
			/*------------------------------------------------------------------------------------
			// Load Skinning Engine 
			-------------------------------------------------------------------------------------*/
			if( $this->theme_supports( 'primary', 'skins' ) )
				require( $this->theme_framework_dir . '/skins/skinner-init.php');
			
			/*-----------------------------------------------------------------------------------
			// Initialize Custom Post Types
			 -----------------------------------------------------------------------------------*/
			
			/* Load Portfolio*/
			if( $this->theme_supports( 'primary', 'portfolio' ) ){
			
				include( $this->theme_cpt_dir . '/portfolio/portfolio-init.php' );
				include( $this->theme_cpt_dir . '/portfolio/portfolio-template.php' );
				
				include( $this->theme_widgets_dir . '/widget-portfolio.php' );
				
			}
			
			/*------------------------------------------------------------------------------------
			// Load Widgets
			------------------------------------------------------------------------------------*/
				
			if( $this->theme_supports( 'primary', 'widgets' ) ){
			
				include_once($this->theme_framework_dir . '/sidebars/sidebars.php' ); // bootstrap the radium sidebar manager
			
				include( $this->theme_widgets_dir . '/widget-init.php' ); //Load Default Widget Areas
				
				/* include widgets*/
				include( $this->theme_widgets_dir . '/widget-custom-images.php' );		
				include( $this->theme_widgets_dir . '/widget-flickr.php' );
				include( $this->theme_widgets_dir . '/widget-menu.php' );
				include( $this->theme_widgets_dir . '/widget-recent-posts.php' );
			 	include( $this->theme_widgets_dir . '/widget-social-counter.php' );
			  	include( $this->theme_widgets_dir . '/widget-social-media.php' );
			  	include( $this->theme_widgets_dir . '/widget-tabbed-content.php' );
				include( $this->theme_widgets_dir . '/widget-twitter.php' );
			
			}

			// We've separated admin and frontend specific files for the best performance
			if( is_admin() ) { 
				
				// Load up our theme options page and related code.
				require( $this->theme_framework_dir . '/options/options-init.php' ); //load admin theme options panel
				require( $this->theme_functions_dir . '/theme-options.php' );// load theme options
				
				require_once( $this->theme_framework_dir . '/extensions/plugin-activation.php' ); // Plugin Activation Dependencies
				require_once( $this->theme_framework_dir . '/extensions/presstrends.php' ); // Presstrends Api
				
				/*------------------------------------------------------------------------------------
				// Load the themes meta fields
				 ------------------------------------------------------------------------------------*/
				if( $this->theme_supports( 'primary', 'meta' ) ) {
					include( $this->theme_includes_dir . '/metaboxes/meta-init.php');
					
					include( $this->theme_includes_dir . '/metaboxes/meta-page.php');
					include( $this->theme_includes_dir . '/metaboxes/meta-post.php');
				}
				
				if( $this->theme_supports( 'primary', 'portfolio' ) && $this->theme_supports( 'primary', 'meta' ) )
					include( $this->theme_includes_dir . '/metaboxes/meta-portfolio.php');
					
				if( $this->theme_supports( 'primary', 'sliders' ) && $this->theme_supports( 'primary', 'meta' ) )
					include( $this->theme_includes_dir . '/metaboxes/meta-slider.php');
				
				/*------------------------------------------------------------------------------------
				// Load the Form Builder
				 -------------------------------------------------------------------------------------*/
		 		if( $this->theme_supports( 'primary', 'forms' ) )
		 			include($this->theme_forms_dir . '/admin.php');
		 		
		 		/*------------------------------------------------------------------------------------
		 		// Load the Slider Engines
		 		 ------------------------------------------------------------------------------------*/
		 		if( $this->theme_supports( 'primary', 'sliders' ) )
		 			include( $this->theme_framework_dir . '/slider/slider-admin.php' );
		 		
			} else {
				
				/* Load Navigation Tools */
				if( $this->theme_supports( 'primary', 'responsive' ) )
					include( $this->theme_functions_dir . '/navigation/mobile-nav.php' ); 
				
				if( $this->theme_supports( 'primary', 'portfolio' ) ) 
					include( $this->theme_functions_dir . '/navigation/portfolio.php' );
				
				/* Pagination */
				include( $this->theme_functions_dir . '/navigation/pagination.php' );
				
				/* Comments */
				include( $this->theme_functions_dir . '/comments.php' );
				
				/* Media - video, audio, image functions */
				require( $this->theme_functions_dir . '/media.php' );
			   	
		 	}
			  	
		}
					
	} //end class
	
	/**
	 * The main function responsible for returning the one true radium framework Instance
	 * to functions everywhere.
	 *
	 * Use this function like you would a global variable, except without needing
	 * to declare the global.
	 *
	 * Example: <?php $framework = radium_framework(); ?>
	 *
	 * @return The one true radium framework Instance
	 */
	function radium_framework() {
		return Radium_Theme_Framework::instance();
	}
	radium_framework(); //All systems go

endif; // class_exists check